"""SQLAlchemy ORM Models for Aera"""
from sqlalchemy import Column, String, Integer, Boolean, ForeignKey, DateTime, Text, JSON
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func
import uuid
from database import Base

def generate_uuid():
    return str(uuid.uuid4())

class User(Base):
    __tablename__ = "users"

    id = Column(String(36), primary_key=True, default=generate_uuid)
    email = Column(String(255), unique=True, index=True, nullable=False)
    hashed_password = Column(String(255), nullable=False)
    full_name = Column(String(255), nullable=False)
    role = Column(String(50), default="Reviewer")  # SuperAdmin, Reviewer, Auditor
    is_active = Column(Boolean, default=True)
    created_at = Column(DateTime(timezone=True), server_default=func.now())

class Conversation(Base):
    __tablename__ = "conversations"

    id = Column(String(36), primary_key=True, default=generate_uuid)
    teams_thread_id = Column(String(255), unique=True, index=True, nullable=False)
    participants = Column(JSON)  # Stores List[Dict] like [{"name": "X", "email": "Y"}]
    topic = Column(String(255), nullable=True)
    last_synced_at = Column(DateTime(timezone=True), server_default=func.now())
    created_at = Column(DateTime(timezone=True), server_default=func.now())

    # Relationships
    messages = relationship("Message", back_populates="conversation", cascade="all, delete-orphan")
    risk_analysis = relationship("RiskAnalysis", back_populates="conversation", uselist=False, cascade="all, delete-orphan")

class Message(Base):
    __tablename__ = "messages"

    id = Column(String(36), primary_key=True, default=generate_uuid)
    conversation_id = Column(String(36), ForeignKey("conversations.id"), nullable=False)
    sender_email = Column(String(255), nullable=False)
    sender_name = Column(String(255), nullable=False)
    content = Column(Text, nullable=False)
    timestamp = Column(DateTime(timezone=True), server_default=func.now())

    conversation = relationship("Conversation", back_populates="messages")

class RiskAnalysis(Base):
    __tablename__ = "risk_analyses"

    id = Column(String(36), primary_key=True, default=generate_uuid)
    conversation_id = Column(String(36), ForeignKey("conversations.id"), nullable=False)
    risk_score = Column(Integer, default=0)
    risk_level = Column(String(50), default="LOW")  # LOW, MEDIUM, HIGH
    ai_explanation = Column(Text, nullable=True)
    risk_indicators = Column(JSON)  # List[str]
    cross_chat_context = Column(Text, nullable=True)
    analyzed_at = Column(DateTime(timezone=True), server_default=func.now())
    status = Column(String(50), default="pending")  # pending, reviewed, dismissed

    conversation = relationship("Conversation", back_populates="risk_analysis")
    review_actions = relationship("ReviewAction", back_populates="risk_analysis", cascade="all, delete-orphan")

class ReviewAction(Base):
    __tablename__ = "review_actions"

    id = Column(String(36), primary_key=True, default=generate_uuid)
    risk_analysis_id = Column(String(36), ForeignKey("risk_analyses.id"), nullable=False)
    reviewer_id = Column(String(36), ForeignKey("users.id"), nullable=False)
    reviewer_name = Column(String(255), nullable=False)
    action = Column(String(50), nullable=False)  # Dismiss, Monitor, Escalate
    notes = Column(Text, nullable=True)
    timestamp = Column(DateTime(timezone=True), server_default=func.now())

    risk_analysis = relationship("RiskAnalysis", back_populates="review_actions")
    # Optional: Relationship to User if needed
    # reviewer = relationship("User") 

class AuditLog(Base):
    __tablename__ = "audit_logs"

    id = Column(String(36), primary_key=True, default=generate_uuid)
    actor_id = Column(String(36), nullable=False)
    actor_email = Column(String(255), nullable=False)
    action = Column(String(50), nullable=False)
    details = Column(JSON)
    timestamp = Column(DateTime(timezone=True), server_default=func.now())