"""Audit Router - System Logs (MySQL/SQLAlchemy)"""
from fastapi import APIRouter, Depends, Query, HTTPException
from typing import List, Optional
from sqlalchemy import select, desc
from sqlalchemy.ext.asyncio import AsyncSession

from models import AuditLog, User
from auth import get_current_active_user
from database import get_db

# FIXED: Removed 'prefix="/audit"' to avoid double prefixing.
# server.py already mounts this at "/api/audit"
router = APIRouter(tags=["Audit"])

# FIXED: Added "/logs" to the path so it matches /api/audit/logs
@router.get("/logs", response_model=List[dict])
async def get_audit_logs(
    limit: int = Query(50, ge=1, le=200),
    action_type: Optional[str] = None,
    current_user: User = Depends(get_current_active_user),
    db: AsyncSession = Depends(get_db)
):
    """Get system audit logs"""
    try:
        stmt = select(AuditLog).order_by(desc(AuditLog.timestamp)).limit(limit)
        
        if action_type:
            stmt = stmt.where(AuditLog.action == action_type)
            
        result = await db.execute(stmt)
        logs = result.scalars().all()
        
        # Manual serialization for safety with JSON fields
        serialized_logs = []
        for log in logs:
            serialized_logs.append({
                "id": log.id,
                "actor_id": log.actor_id,
                "actor_email": log.actor_email,
                "action": log.action,
                "details": log.details if log.details else {},
                "timestamp": log.timestamp
            })
            
        return serialized_logs
        
    except Exception as e:
        # Log the error internally
        print(f"Audit Log Error: {e}")
        raise HTTPException(status_code=500, detail="Failed to fetch audit logs")