"""Chatbot Router - RAG-Lite Querying (MySQL/SQLAlchemy)"""
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy import select, or_, desc
from sqlalchemy.ext.asyncio import AsyncSession

from models import RiskAnalysis, Conversation, User
from schemas import ChatbotQuery, ChatbotResponse
from auth import get_current_active_user
from database import get_db
from services.claude_service import claude_service

router = APIRouter(prefix="/chatbot", tags=["Chatbot"])

@router.post("/query", response_model=ChatbotResponse)
async def query_chatbot(
    query_data: ChatbotQuery,
    current_user: User = Depends(get_current_active_user),
    db: AsyncSession = Depends(get_db)
):
    """
    Query the compliance knowledge base.
    This "RAG-Lite" implementation searches recent high-risk cases 
    and feeds them to Claude to answer the user's question.
    """
    
    # 1. Retrieval: Find relevant high/medium risk analyses
    # In a full prod system, you'd use vector embeddings here.
    # For POC, we fetch recent flagged items.
    stmt = select(RiskAnalysis).where(
        RiskAnalysis.risk_level.in_(["HIGH", "MEDIUM"])
    ).order_by(desc(RiskAnalysis.analyzed_at)).limit(10)
    
    result = await db.execute(stmt)
    relevant_risks = result.scalars().all()
    
    # 2. Context Construction
    context = "Here are the recent flagged compliance cases:\n"
    sources = []
    
    for risk in relevant_risks:
        context += f"- Case ID {risk.id}: Risk Level {risk.risk_level}. Reason: {risk.ai_explanation}\n"
        sources.append({
            "id": risk.id,
            "level": risk.risk_level,
            "reason": risk.ai_explanation
        })
        
    # 3. AI Synthesis (We reuse the raw Anthropic client from service)
    prompt = f"""
    You are an AI assistant for the Aera Compliance System.
    
    USER QUERY: "{query_data.query}"
    
    SYSTEM DATA (Recent Risks):
    {context}
    
    INSTRUCTIONS:
    - Answer the user's question based ONLY on the system data provided.
    - If the answer isn't in the data, say "I don't see any current records matching that inquiry."
    - Be professional and concise.
    """
    
    try:
        response = await claude_service.client.messages.create(
            model=claude_service.model,
            max_tokens=300,
            messages=[{"role": "user", "content": prompt}]
        )
        answer = response.content[0].text
    except Exception:
        answer = "I'm having trouble connecting to the AI engine right now."

    return ChatbotResponse(
        response=answer,
        sources=sources
    )