"""Risk Analysis Router - Flagged Messages and Reviews (MySQL/SQLAlchemy)"""
from fastapi import APIRouter, Depends, HTTPException, status, Query
from typing import List, Optional
from datetime import datetime
from sqlalchemy import select, desc
from sqlalchemy.orm import selectinload
from sqlalchemy.ext.asyncio import AsyncSession

# Import Models and Schemas
from models import RiskAnalysis, Conversation, ReviewAction, AuditLog, User
from schemas import ReviewActionCreate, ReviewActionEnum

# Dependencies
from auth import get_current_active_user
from database import get_db

router = APIRouter(prefix="/risks", tags=["Risk Analysis"])

@router.get("", response_model=List[dict])
async def get_flagged_risks(
    risk_level: Optional[str] = Query(None, description="Filter by risk level: HIGH, MEDIUM, LOW"),
    status_filter: Optional[str] = Query(None, alias="status", description="Filter by status"),
    limit: int = Query(50, ge=1, le=100),
    current_user: User = Depends(get_current_active_user),
    db: AsyncSession = Depends(get_db)
):
    """Get all flagged risk analyses"""
    stmt = select(RiskAnalysis).options(
        selectinload(RiskAnalysis.conversation).selectinload(Conversation.messages),
        selectinload(RiskAnalysis.review_actions)
    ).order_by(desc(RiskAnalysis.analyzed_at)).limit(limit)
    
    if risk_level:
        stmt = stmt.where(RiskAnalysis.risk_level == risk_level.upper())
    
    if status_filter:
        stmt = stmt.where(RiskAnalysis.status == status_filter.lower())
    
    result = await db.execute(stmt)
    risks = result.scalars().all()
    
    enriched = []
    for r in risks:
        risk_dict = {
            "id": r.id,
            "conversation_id": r.conversation_id,
            "risk_score": r.risk_score,
            "risk_level": r.risk_level,
            "ai_explanation": r.ai_explanation,
            "status": r.status,
            "analyzed_at": r.analyzed_at,
            "conversation": None,
            "review_actions": []
        }
        
        if r.conversation:
            # FIX: Sort messages by timestamp explicitly (Oldest -> Newest)
            sorted_messages = sorted(r.conversation.messages, key=lambda m: m.timestamp)
            
            risk_dict["conversation"] = {
                "id": r.conversation.id,
                "participants": r.conversation.participants,
                "topic": r.conversation.topic,
                "messages": [
                    {
                        "sender_name": m.sender_name,
                        "content": m.content,
                        "timestamp": m.timestamp
                    } for m in sorted_messages
                ]
            }
            
        if r.review_actions:
            risk_dict["review_actions"] = [
                {
                    "action": ra.action,
                    "reviewer_name": ra.reviewer_name,
                    "notes": ra.notes,
                    "timestamp": ra.timestamp
                } for ra in r.review_actions
            ]
            
        enriched.append(risk_dict)
    
    return enriched

@router.get("/{risk_id}")
async def get_risk_detail(
    risk_id: str,
    current_user: User = Depends(get_current_active_user),
    db: AsyncSession = Depends(get_db)
):
    """Get detailed risk analysis with conversation"""
    stmt = select(RiskAnalysis).options(
        selectinload(RiskAnalysis.conversation).selectinload(Conversation.messages),
        selectinload(RiskAnalysis.review_actions)
    ).where(RiskAnalysis.id == risk_id)
    
    result = await db.execute(stmt)
    risk = result.scalars().first()
    
    if not risk:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Risk analysis not found"
        )
    
    risk_dict = {
        "id": risk.id,
        "risk_level": risk.risk_level,
        "risk_score": risk.risk_score,
        "ai_explanation": risk.ai_explanation,
        "status": risk.status,
        "analyzed_at": risk.analyzed_at,
        "cross_chat_context": risk.cross_chat_context,
        "conversation": None,
        "review_actions": []
    }
    
    if risk.conversation:
        # FIX: Sort messages by timestamp explicitly (Oldest -> Newest)
        sorted_messages = sorted(risk.conversation.messages, key=lambda m: m.timestamp)

        risk_dict["conversation"] = {
            "id": risk.conversation.id,
            "topic": risk.conversation.topic,
            "participants": risk.conversation.participants,
            "messages": [
                {
                    "sender_name": m.sender_name,
                    "content": m.content,
                    "timestamp": m.timestamp,
                    "sender_email": m.sender_email
                } for m in sorted_messages
            ]
        }
    
    if risk.review_actions:
        risk_dict["review_actions"] = [
            {"reviewer_name": ra.reviewer_name, "action": ra.action, "notes": ra.notes, "timestamp": ra.timestamp}
            for ra in risk.review_actions
        ]

    return risk_dict

@router.post("/{risk_id}/review")
async def submit_review_action(
    risk_id: str,
    action_data: ReviewActionCreate,
    current_user: User = Depends(get_current_active_user),
    db: AsyncSession = Depends(get_db)
):
    """Submit a review action (Dismiss, Monitor, Escalate)"""
    # 1. Verify risk exists
    result = await db.execute(select(RiskAnalysis).where(RiskAnalysis.id == risk_id))
    risk = result.scalars().first()
    
    if not risk:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Risk analysis not found"
        )

    # 2. Fetch Conversation details for the Audit Log
    stmt_conv = select(Conversation).where(Conversation.id == risk.conversation_id)
    res_conv = await db.execute(stmt_conv)
    conversation = res_conv.scalars().first()
    
    # Extract Human-Readable Details
    topic = conversation.topic if conversation else "Unknown Chat"
    participants = []
    if conversation and conversation.participants:
        participants = [p.get("name", "Unknown") for p in conversation.participants]

    # 3. Create Review Action
    new_action = ReviewAction(
        risk_analysis_id=risk_id,
        reviewer_id=current_user.id,
        reviewer_name=current_user.full_name,
        action=action_data.action.value, 
        notes=action_data.notes,
        timestamp=datetime.utcnow()
    )
    db.add(new_action)
    
    # 4. Update Status
    new_status = "reviewed"
    if action_data.action == ReviewActionEnum.ESCALATE:
        new_status = "escalated"
    elif action_data.action == ReviewActionEnum.DISMISS:
        new_status = "dismissed"
    
    risk.status = new_status
    
    # 5. Create Enriched Audit Log
    action_display_name = action_data.action.value.title()
    
    audit_log = AuditLog(
        actor_id=current_user.id,
        actor_email=current_user.email,
        action=f"Case {action_display_name}d", 
        details={
            "risk_id": risk_id,
            "action": action_data.action.value,
            "notes": action_data.notes,
            "chat_topic": topic,
            "chat_participants": participants
        },
        timestamp=datetime.utcnow()
    )
    db.add(audit_log)
    
    await db.commit()
    
    return {"success": True, "message": f"Action '{action_data.action.value}' recorded successfully"}

@router.get("/{risk_id}/actions", response_model=List[dict])
async def get_review_actions(
    risk_id: str,
    current_user: User = Depends(get_current_active_user),
    db: AsyncSession = Depends(get_db)
):
    """Get all review actions for a risk"""
    stmt = select(ReviewAction).where(ReviewAction.risk_analysis_id == risk_id).order_by(desc(ReviewAction.timestamp))
    result = await db.execute(stmt)
    actions = result.scalars().all()
    return actions