"""Sync Router - Trigger data ingestion and analysis (MySQL/SQLAlchemy)"""
from fastapi import APIRouter, Depends
from sqlalchemy.ext.asyncio import AsyncSession

# Import Schema for Response
from schemas import SyncResponse
# Import Model for User type hinting
from models import User
# Import Auth dependency
from auth import get_current_active_user
# Import DB dependency
from database import get_db
# Import Service
from services.sync_service import sync_service

router = APIRouter(prefix="/sync", tags=["Sync"])

@router.post("", response_model=SyncResponse)
async def trigger_sync(
    current_user: User = Depends(get_current_active_user),
    db: AsyncSession = Depends(get_db)
):
    """
    Trigger a sync operation:
    1. Fetch conversations from MS Teams (or mock data)
    2. Analyze each conversation with Claude AI
    3. Flag medium and high-risk conversations
    """
    # Note: We pass the 'db' session to the service layer so it can save results
    result = await sync_service.sync_and_analyze(
        db=db,
        user_id=current_user.id,
        user_email=current_user.email
    )
    
    return SyncResponse(
        success=result["success"],
        conversations_synced=result["conversations_synced"],
        risks_detected=result["risks_detected"],
        message=result["message"]
    )