"""Pydantic Schemas for API Data Transfer (MySQL Version)"""
from pydantic import BaseModel, Field, ConfigDict, EmailStr
from typing import Optional, List, Dict, Any
from datetime import datetime
from enum import Enum

# --- ENUMS ---
class UserRole(str, Enum):
    SUPER_ADMIN = "SuperAdmin"
    REVIEWER = "Reviewer"
    AUDITOR = "Auditor"

class RiskLevel(str, Enum):
    LOW = "LOW"
    MEDIUM = "MEDIUM"
    HIGH = "HIGH"

class ReviewActionEnum(str, Enum):
    DISMISS = "Dismiss"
    MONITOR = "Monitor"
    ESCALATE = "Escalate"

# --- USER SCHEMAS ---
class UserBase(BaseModel):
    email: EmailStr
    full_name: str
    role: UserRole = UserRole.REVIEWER

class UserCreate(UserBase):
    password: str

class UserLogin(BaseModel):
    email: EmailStr
    password: str

class UserResponse(UserBase):
    id: str
    is_active: bool
    created_at: datetime
    
    model_config = ConfigDict(from_attributes=True)

# --- TOKEN SCHEMAS ---
class Token(BaseModel):
    access_token: str
    refresh_token: str
    token_type: str = "bearer"

class TokenData(BaseModel):
    user_id: Optional[str] = None
    email: Optional[str] = None
    role: Optional[str] = None

class RefreshTokenRequest(BaseModel):
    refresh_token: str

# --- SYNC SCHEMAS ---
class SyncResponse(BaseModel):
    success: bool
    conversations_synced: int
    risks_detected: int
    message: str

# --- CONVERSATION & MESSAGE SCHEMAS ---
class MessageBase(BaseModel):
    sender_email: str
    sender_name: str
    content: str
    timestamp: datetime

class MessageResponse(MessageBase):
    id: str
    conversation_id: str
    model_config = ConfigDict(from_attributes=True)

class ConversationBase(BaseModel):
    teams_thread_id: str
    participants: List[Dict[str, str]]
    topic: Optional[str] = None

class ConversationResponse(ConversationBase):
    id: str
    last_synced_at: datetime
    created_at: datetime
    messages: List[MessageResponse] = []
    
    model_config = ConfigDict(from_attributes=True)

# --- RISK ANALYSIS SCHEMAS ---
class RiskAnalysisBase(BaseModel):
    risk_score: int
    risk_level: str  # String to handle both Enum and DB string
    ai_explanation: str
    risk_indicators: List[str] = []
    cross_chat_context: Optional[str] = None

class RiskAnalysisResponse(RiskAnalysisBase):
    id: str
    conversation_id: str
    analyzed_at: datetime
    status: str
    model_config = ConfigDict(from_attributes=True)

# --- DASHBOARD SCHEMAS ---
class DashboardStats(BaseModel):
    total_conversations: int
    total_flagged: int
    high_risk_count: int
    medium_risk_count: int
    low_risk_count: int
    pending_reviews: int
    reviewed_today: int

class RiskTrendItem(BaseModel):
    date: str
    high: int
    medium: int
    low: int

# --- ACTION & AUDIT SCHEMAS ---
class ReviewActionCreate(BaseModel):
    risk_analysis_id: str
    action: ReviewActionEnum
    notes: Optional[str] = None

class AuditLogResponse(BaseModel):
    id: str
    actor_email: str
    action: str
    details: Dict[str, Any]
    timestamp: datetime
    model_config = ConfigDict(from_attributes=True)

# --- CHATBOT SCHEMAS ---
class ChatbotQuery(BaseModel):
    query: str

class ChatbotResponse(BaseModel):
    response: str
    sources: List[Dict[str, Any]] = []