"""Main FastAPI Application for Aera Risk Detection System (MySQL Edition)"""
from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
import logging
from contextlib import asynccontextmanager

from config import settings
from database import engine
from models import Base
from routers import auth, sync, dashboard, risks, chatbot, audit

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

@asynccontextmanager
async def lifespan(app: FastAPI):
    # --- Startup ---
    logger.info("Starting Aera Risk Detection System...")
    logger.info(f"Mock Mode: {settings.MOCK_MODE}")
    
    # Create Database Tables if they don't exist
    async with engine.begin() as conn:
        await conn.run_sync(Base.metadata.create_all)
    logger.info("Database tables verified.")
    
    yield
    
    # --- Shutdown ---
    logger.info("Shutting down...")
    await engine.dispose()

app = FastAPI(
    title="Aera Risk Detection API",
    description="Centralized Risk Detection for Microsoft Teams",
    version="1.0.0",
    lifespan=lifespan
)

# CORS
app.add_middleware(
    CORSMiddleware,
    allow_origins=settings.CORS_ORIGINS.split(",") if settings.CORS_ORIGINS != "*" else ["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Include routers
# Standard routers usually have their own internal prefix (e.g. /auth)
app.include_router(auth.router, prefix="/api")
app.include_router(sync.router, prefix="/api")
app.include_router(dashboard.router, prefix="/api")
app.include_router(risks.router, prefix="/api")
app.include_router(chatbot.router, prefix="/api")

# FIXED: Mount audit router explicitly at /api/audit
# This combines with the router's "/logs" to create "/api/audit/logs"
app.include_router(audit.router, prefix="/api/audit")

@app.get("/api/health")
async def health_check():
    return {
        "status": "healthy",
        "service": "Aera Risk Detection",
        "mock_mode": settings.MOCK_MODE,
        "database": "MySQL"
    }

@app.get("/api/")
async def root():
    return {
        "message": "Welcome to Aera Risk Detection API",
        "version": "1.0.0",
        "docs": "/docs"
    }