"""Claude AI Service for Risk Detection"""
import json
import logging
from typing import Dict, Any, List, Optional
from anthropic import AsyncAnthropic
from config import settings
from schemas import RiskLevel

logger = logging.getLogger(__name__)

class ClaudeService:
    """Service for AI-powered risk detection using Claude"""
    
    def __init__(self):
        self.client = AsyncAnthropic(api_key=settings.ANTHROPIC_API_KEY)
        # FIXED: Using the stable Claude 3.0 Sonnet model
        self.model = "claude-3-haiku-20240307"
    
    async def analyze_conversation(
        self,
        conversation_messages: List[Dict[str, Any]],
        participants: List[Dict[str, str]],
        cross_chat_context: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Analyze a conversation for compliance risks.
        Returns risk_level, risk_score, explanation, and indicators.
        """
        # Format the conversation for analysis
        formatted_messages = self._format_messages(conversation_messages)
        
        # Safe extraction of participant names
        p_names = []
        for p in participants:
            if isinstance(p, dict):
                p_names.append(p.get("name", p.get("email", "Unknown")))
        participants_str = ", ".join(p_names)
        
        # Build the prompt
        system_prompt = """You are an expert corporate compliance and risk detection analyst. Your role is to analyze Microsoft Teams conversations for potential policy violations, unethical behavior, and corporate risks.

RISK CATEGORIES TO DETECT:
1. CONFIDENTIAL DATA LEAKAGE - Sharing sensitive company information externally
2. BRIBERY/KICKBACKS - Discussions of improper payments or incentives
3. FRAUD/DECEPTION - Lying to management, falsifying reports, covering up mistakes
4. INSIDER TRADING - Sharing material non-public information
5. POLICY VIOLATIONS - Bypassing approvals, misusing systems/credentials
6. HARASSMENT/MISCONDUCT - Inappropriate workplace behavior
7. CONFLICT OF INTEREST - Personal interests affecting business decisions
8. CROSS-CHAT CONTRADICTIONS - Employee telling different versions to different people

CLASSIFICATION:
- LOW RISK (0-30): Normal business communication, no concerns
- MEDIUM RISK (31-70): Possible policy concern, review recommended
- HIGH RISK (71-100): Strong indicators of violation, immediate attention needed

IMPORTANT: Be thorough but avoid false positives. Normal workplace conversations should be LOW risk."""

        user_prompt = f"""Analyze this Microsoft Teams conversation between: {participants_str}

CONVERSATION:
{formatted_messages}
"""
        
        if cross_chat_context:
            user_prompt += f"""
IMPORTANT CROSS-CHAT CONTEXT:
This employee has been involved in other recent conversations. Here is a summary:
{cross_chat_context}

Pay special attention to any CONTRADICTIONS between what this employee said in this conversation versus their other conversations. Contradictions are HIGH RISK indicators.
"""
        
        user_prompt += """
Respond with a JSON object (no markdown formatting):
{
    "risk_level": "LOW" | "MEDIUM" | "HIGH",
    "risk_score": <number 0-100>,
    "explanation": "<detailed explanation of findings>",
    "risk_indicators": ["<list of specific risk indicators found>"],
    "key_quotes": ["<relevant quotes from the conversation>"]
}"""

        try:
            response = await self.client.messages.create(
                model=self.model,
                max_tokens=1500,
                messages=[
                    {"role": "user", "content": user_prompt}
                ],
                system=system_prompt
            )
            
            # Parse the response
            response_text = response.content[0].text.strip()
            
            try:
                import re
                if "```" in response_text:
                    match = re.search(r"\{.*\}", response_text, re.DOTALL)
                    if match:
                        response_text = match.group(0)
                
                result = json.loads(response_text)
                
                r_level_str = result.get("risk_level", "LOW").upper()
                if r_level_str not in ["LOW", "MEDIUM", "HIGH"]:
                    r_level_str = "LOW"

                return {
                    "risk_level": RiskLevel(r_level_str),
                    "risk_score": min(100, max(0, int(result.get("risk_score", 0)))),
                    "ai_explanation": result.get("explanation", "Analysis completed."),
                    "risk_indicators": result.get("risk_indicators", []),
                    "key_quotes": result.get("key_quotes", [])
                }
            except json.JSONDecodeError:
                logger.error(f"Failed to parse Claude response: {response_text}")
                return self._default_response("Failed to parse AI response JSON")
                
        except Exception as e:
            logger.error(f"Claude API error: {str(e)}")
            return self._default_response(f"AI analysis error: {str(e)}")
    
    async def generate_cross_chat_summary(
        self,
        employee_email: str,
        recent_conversations: List[Dict[str, Any]]
    ) -> str:
        """Generate a summary of an employee's recent conversations."""
        if not recent_conversations:
            return ""
        
        context_text = ""
        for conv in recent_conversations:
            msgs = conv.get("messages", [])
            msg_preview = "\n".join([f"- {m.get('sender', 'User')}: {m.get('content', '')}" for m in msgs])
            context_text += f"\n[Thread Topic: {conv.get('topic', 'Unknown')}]\n{msg_preview}\n"

        prompt = f"""
        Review these recent chat snippets involving {employee_email}.
        Summarize their key activities, frustrations, or conflicting statements.
        Keep it under 100 words.
        
        CHATS:
        {context_text}
        """

        try:
            response = await self.client.messages.create(
                model=self.model,
                max_tokens=200,
                messages=[{"role": "user", "content": prompt}]
            )
            return response.content[0].text
        except Exception as e:
            logger.error(f"Error generating summary: {e}")
            return "Unable to generate cross-chat summary."
    
    def _format_messages(self, messages: List[Dict[str, Any]]) -> str:
        formatted = []
        for msg in messages:
            sender = msg.get("sender_name", msg.get("sender_email", "Unknown"))
            content = msg.get("content", "")
            timestamp = msg.get("timestamp", "")
            formatted.append(f"[{timestamp}] {sender}: {content}")
        return "\n".join(formatted)
    
    def _default_response(self, error_msg: str) -> Dict[str, Any]:
        return {
            "risk_level": RiskLevel.LOW,
            "risk_score": 0,
            "ai_explanation": error_msg,
            "risk_indicators": [],
            "key_quotes": []
        }

claude_service = ClaudeService()