"""Graph Service - Fetches Real or Mock Data"""
import logging
import msal
import requests
import asyncio
from datetime import datetime
from bs4 import BeautifulSoup
from typing import List, Dict, Any
from config import settings

logger = logging.getLogger(__name__)

class GraphService:
    def __init__(self):
        self.mock_mode = settings.MOCK_MODE
        self.scope = ["https://graph.microsoft.com/.default"]
        self.authority = f"https://login.microsoftonline.com/{settings.AZURE_TENANT_ID}"

    async def fetch_conversations_and_messages(self) -> List[Dict[str, Any]]:
        """
        Main Entry Point:
        - If MOCK_MODE: Returns fake data.
        - If REAL: Authenticates, fetches all messages, groups them by Chat ID.
        """
        if self.mock_mode:
            logger.info("📡 MOCK MODE: Returning simulated Teams data.")
            return self._get_mock_data()
        
        logger.info("🌍 REAL MODE: Connecting to Microsoft Graph...")
        # Run blocking HTTP calls in a thread executor
        loop = asyncio.get_event_loop()
        return await loop.run_in_executor(None, self._fetch_real_data_sync)

    def _fetch_real_data_sync(self) -> List[Dict[str, Any]]:
        """Synchronous implementation of the graph fetching logic"""
        try:
            token = self._get_access_token()
            if not token:
                raise Exception("Failed to acquire access token")

            # 1. Get Users
            users = self._make_graph_call("https://graph.microsoft.com/v1.0/users", token)
            logger.info(f"Found {len(users)} users in tenant.")

            all_raw_messages = []
            
            # NEW: Track Message IDs to prevent duplicates when fetching from multiple sides
            seen_message_ids = set()

            # 2. Get Messages for each user
            for user in users:
                user_id = user['id']
                logger.info(f"Fetching messages for user: {user.get('displayName')}...")
                
                msgs = self._make_graph_call(
                    f"https://graph.microsoft.com/v1.0/users/{user_id}/chats/getAllMessages", 
                    token
                )
                
                # NEW: Filter duplicates immediately
                for m in msgs:
                    msg_id = m.get('id')
                    if msg_id and msg_id not in seen_message_ids:
                        all_raw_messages.append(m)
                        seen_message_ids.add(msg_id)
            
            logger.info(f"Collected {len(all_raw_messages)} unique messages.")

            # 3. Group by Conversation ID
            grouped_chats = {}
            for msg in all_raw_messages:
                if msg.get('messageType') != 'message' or not msg.get('body', {}).get('content'):
                    continue
                
                chat_id = msg.get('chatId')
                if chat_id not in grouped_chats:
                    grouped_chats[chat_id] = []
                grouped_chats[chat_id].append(msg)
            
            # 4. Format for SyncService
            structured_conversations = []
            
            for chat_id, raw_msgs in grouped_chats.items():
                # Sort messages by time
                raw_msgs.sort(key=lambda x: x['createdDateTime'])
                
                senders = set()
                formatted_messages = []
                
                for m in raw_msgs:
                    sender_name = m.get('from', {}).get('user', {}).get('displayName', 'Unknown')
                    sender_email = f"{sender_name.replace(' ', '.').lower()}@company.com"
                    
                    if sender_name != 'Unknown':
                        senders.add(f"{sender_name}|{sender_email}")
                    
                    clean_text = self._clean_html(m.get('body', {}).get('content', ''))
                    
                    formatted_messages.append({
                        "id": m.get('id'),
                        "sender_name": sender_name,
                        "sender_email": sender_email,
                        "content": clean_text,
                        "created_at": m.get('createdDateTime') # ISO string
                    })

                # Format Participants
                participants_list = []
                for s in senders:
                    name, email = s.split('|')
                    participants_list.append({"name": name, "email": email})

                structured_conversations.append({
                    "id": chat_id,
                    "topic": "Teams Chat", 
                    "participants": participants_list,
                    "messages": formatted_messages
                })
            
            logger.info(f"✅ Successfully processed {len(structured_conversations)} conversations from Graph.")
            return structured_conversations

        except Exception as e:
            logger.error(f"Graph API Failed: {str(e)}")
            raise e

    # --- HELPERS ---
    def _get_access_token(self):
        app = msal.ConfidentialClientApplication(
            settings.AZURE_CLIENT_ID,
            authority=self.authority,
            client_credential=settings.AZURE_CLIENT_SECRET
        )
        result = app.acquire_token_for_client(scopes=self.scope)
        if "access_token" in result:
            return result["access_token"]
        else:
            logger.error(f"Auth Error: {result.get('error_description')}")
            return None

    def _make_graph_call(self, url, token):
        headers = {'Authorization': f'Bearer {token}'}
        results = []
        while url:
            try:
                response = requests.get(url, headers=headers)
                if response.status_code == 429:
                    import time
                    time.sleep(2) 
                    continue
                response.raise_for_status()
                data = response.json()
                if 'value' in data:
                    results.extend(data['value'])
                url = data.get('@odata.nextLink', None)
            except Exception as e:
                logger.error(f"Graph Request Error: {e}")
                break
        return results

    def _clean_html(self, raw_html):
        if not raw_html: return ""
        return BeautifulSoup(raw_html, "html.parser").get_text(separator=" ").strip()

    # --- MOCK DATA ---
    def _get_mock_data(self):
        now_iso = datetime.utcnow().isoformat()
        return [
            {
                "id": "thread_contradiction_1",
                "topic": "Project Status Update",
                "participants": [{"name": "Aniket Sharma", "email": "aniket@company.com"}, {"name": "Manager", "email": "manager@company.com"}],
                "messages": [
                    {"sender_name": "Aniket Sharma", "sender_email": "aniket@company.com", "content": "Hi Manager, just an update: The database deployment was successful. Zero downtime.", "created_at": now_iso},
                    {"sender_name": "Manager", "sender_email": "manager@company.com", "content": "Great work Aniket. Thanks for the update.", "created_at": now_iso}
                ]
            },
            {
                "id": "thread_contradiction_2",
                "topic": "Quick Help Needed",
                "participants": [{"name": "Aniket Sharma", "email": "aniket@company.com"}, {"name": "Suraj Patel", "email": "suraj@company.com"}],
                "messages": [
                    {"sender_name": "Aniket Sharma", "sender_email": "aniket@company.com", "content": "Bro I messed up. I accidentally deleted the production DB while trying to fix the deployment.", "created_at": now_iso},
                    {"sender_name": "Suraj Patel", "sender_email": "suraj@company.com", "content": "Wait, didn't you just tell the manager it was successful?", "created_at": now_iso},
                    {"sender_name": "Aniket Sharma", "sender_email": "aniket@company.com", "content": "Yeah I panicked. I'm trying to restore it from backup before he notices. Don't say anything.", "created_at": now_iso}
                ]
            },
            {
                "id": "thread_bribery_1",
                "topic": "Vendor Discussion",
                "participants": [{"name": "Priya Mehta", "email": "priya@company.com"}, {"name": "External Vendor", "email": "vendor@external.com"}],
                "messages": [
                    {"sender_name": "Priya Mehta", "sender_email": "priya@company.com", "content": "The contract looks good, but my boss is hesitant about the price.", "created_at": now_iso},
                    {"sender_name": "External Vendor", "sender_email": "vendor@external.com", "content": "I understand. If you can push this through by Friday, we can arrange a 5% consultation fee for you personally.", "created_at": now_iso},
                    {"sender_name": "Priya Mehta", "sender_email": "priya@company.com", "content": "That sounds interesting. Let's discuss the details on a private channel.", "created_at": now_iso}
                ]
            }
        ]

graph_service = GraphService()