import React, { useState } from 'react';
import { Link, useLocation, useNavigate } from 'react-router-dom';
import { useAuth, apiClient } from '../lib/api';
import { toast } from 'sonner';
import {
  LayoutDashboard,
  AlertTriangle,
  FileText,
  Settings,
  LogOut,
  RefreshCw,
  Shield,
  ChevronLeft,
  ChevronRight,
  User,
  MessageSquare,
} from 'lucide-react';
import { Button } from './ui/button';
import { cn } from '../lib/utils';

const navigation = [
  { name: 'Dashboard', href: '/', icon: LayoutDashboard },
  { name: 'Flagged Messages', href: '/flagged', icon: AlertTriangle },
  { name: 'Audit Logs', href: '/audit', icon: FileText, adminOnly: true },
  { name: 'Settings', href: '/settings', icon: Settings },
];

export const Sidebar = () => {
  const { user, logout, isAdmin } = useAuth();
  const location = useLocation();
  const navigate = useNavigate();
  const [collapsed, setCollapsed] = useState(false);
  const [syncing, setSyncing] = useState(false);

  const handleSync = async () => {
    setSyncing(true);
    try {
      const response = await apiClient.triggerSync();
      toast.success(response.data.message);
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Sync failed');
    } finally {
      setSyncing(false);
    }
  };

  const handleLogout = () => {
    logout();
    navigate('/login');
  };

  const filteredNav = navigation.filter(
    (item) => !item.adminOnly || (item.adminOnly && isAdmin)
  );

  return (
    <aside
      data-testid="sidebar"
      className={cn(
        'fixed left-0 top-0 z-40 h-screen bg-slate-900 text-white transition-all duration-300',
        collapsed ? 'w-16' : 'w-64'
      )}
    >
      <div className="flex h-full flex-col">
        {/* Logo */}
        <div className="flex h-16 items-center justify-between border-b border-slate-700 px-4">
          {!collapsed && (
            <div className="flex items-center gap-2">
              <Shield className="h-8 w-8 text-blue-400" />
              <span className="text-xl font-bold tracking-tight font-['Manrope']">
                Aera
              </span>
            </div>
          )}
          {collapsed && <Shield className="h-8 w-8 text-blue-400 mx-auto" />}
          <button
            onClick={() => setCollapsed(!collapsed)}
            className="rounded p-1 hover:bg-slate-800"
            data-testid="sidebar-toggle"
          >
            {collapsed ? (
              <ChevronRight className="h-4 w-4" />
            ) : (
              <ChevronLeft className="h-4 w-4" />
            )}
          </button>
        </div>

        {/* Sync Button */}
        <div className="p-4">
          <Button
            onClick={handleSync}
            disabled={syncing}
            className={cn(
              'w-full bg-blue-600 hover:bg-blue-700 text-white font-medium',
              collapsed && 'p-2'
            )}
            data-testid="sync-button"
          >
            <RefreshCw
              className={cn('h-4 w-4', syncing && 'animate-spin', !collapsed && 'mr-2')}
            />
            {!collapsed && (syncing ? 'Syncing...' : 'Sync Teams')}
          </Button>
        </div>

        {/* Navigation */}
        <nav className="flex-1 space-y-1 px-3">
          {filteredNav.map((item) => {
            const isActive = location.pathname === item.href;
            return (
              <Link
                key={item.name}
                to={item.href}
                data-testid={`nav-${item.name.toLowerCase().replace(' ', '-')}`}
                className={cn(
                  'flex items-center gap-3 rounded-md px-3 py-2 text-sm font-medium transition-colors',
                  isActive
                    ? 'bg-slate-800 text-white'
                    : 'text-slate-300 hover:bg-slate-800 hover:text-white',
                  collapsed && 'justify-center px-2'
                )}
              >
                <item.icon className="h-5 w-5 flex-shrink-0" />
                {!collapsed && <span>{item.name}</span>}
              </Link>
            );
          })}
        </nav>

        {/* User Section */}
        <div className="border-t border-slate-700 p-4">
          <div
            className={cn(
              'flex items-center gap-3',
              collapsed && 'flex-col gap-2'
            )}
          >
            <div className="flex h-9 w-9 items-center justify-center rounded-full bg-slate-700">
              <User className="h-5 w-5 text-slate-300" />
            </div>
            {!collapsed && (
              <div className="flex-1 min-w-0">
                <p className="truncate text-sm font-medium">{user?.full_name}</p>
                <p className="truncate text-xs text-slate-400">{user?.role}</p>
              </div>
            )}
            <button
              onClick={handleLogout}
              className="rounded p-2 hover:bg-slate-800"
              data-testid="logout-button"
              title="Logout"
            >
              <LogOut className="h-4 w-4 text-slate-400" />
            </button>
          </div>
        </div>
      </div>
    </aside>
  );
};

export const DashboardLayout = ({ children, title }) => {
  const [sidebarCollapsed, setSidebarCollapsed] = useState(false);

  return (
    <div className="min-h-screen bg-slate-50">
      <Sidebar />
      <main
        className={cn(
          'transition-all duration-300',
          sidebarCollapsed ? 'ml-16' : 'ml-64'
        )}
        style={{ marginLeft: '256px' }}
      >
        {/* Header */}
        <header className="sticky top-0 z-30 border-b border-slate-200 bg-white/80 backdrop-blur-md">
          <div className="flex h-16 items-center justify-between px-6">
            <h1 className="text-2xl font-semibold tracking-tight text-slate-900 font-['Manrope']">
              {title}
            </h1>
          </div>
        </header>

        {/* Content */}
        <div className="p-6">{children}</div>
      </main>
    </div>
  );
};

export default DashboardLayout;
