import React, { createContext, useContext, useState, useEffect } from 'react';
import axios from 'axios';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

// Create axios instance
const api = axios.create({
  baseURL: API,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Auth Context
const AuthContext = createContext(null);

export const useAuth = () => {
  const context = useContext(AuthContext);
  if (!context) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
};

export const AuthProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [token, setToken] = useState(localStorage.getItem('access_token'));
  const [loading, setLoading] = useState(true);

  // Set up axios interceptor for auth
  useEffect(() => {
    const interceptor = api.interceptors.request.use((config) => {
      const storedToken = localStorage.getItem('access_token');
      if (storedToken) {
        config.headers.Authorization = `Bearer ${storedToken}`;
      }
      return config;
    });

    return () => api.interceptors.request.eject(interceptor);
  }, []);

  // Fetch user on mount if token exists
  useEffect(() => {
    const fetchUser = async () => {
      if (token) {
        try {
          const response = await api.get('/auth/me');
          setUser(response.data);
        } catch (error) {
          console.error('Failed to fetch user:', error);
          logout();
        }
      }
      setLoading(false);
    };

    fetchUser();
  }, [token]);

  const login = async (email, password) => {
    const response = await api.post('/auth/login/json', { email, password });
    const { access_token, refresh_token } = response.data;
    
    localStorage.setItem('access_token', access_token);
    localStorage.setItem('refresh_token', refresh_token);
    setToken(access_token);
    
    const userResponse = await api.get('/auth/me');
    setUser(userResponse.data);
    
    return userResponse.data;
  };

  const register = async (email, password, fullName, role = 'Reviewer') => {
    const response = await api.post('/auth/register', {
      email,
      password,
      full_name: fullName,
      role,
    });
    return response.data;
  };

  const logout = () => {
    localStorage.removeItem('access_token');
    localStorage.removeItem('refresh_token');
    setToken(null);
    setUser(null);
  };

  const value = {
    user,
    token,
    loading,
    login,
    register,
    logout,
    isAuthenticated: !!user,
    isAdmin: user?.role === 'SuperAdmin',
  };

  return <AuthContext.Provider value={value}>{children}</AuthContext.Provider>;
};

// API functions
export const apiClient = {
  // Dashboard
  getStats: () => api.get('/dashboard/stats'),
  getTrends: () => api.get('/dashboard/trends'),
  getRecentAlerts: (limit = 5) => api.get(`/dashboard/recent-alerts?limit=${limit}`),

  // Sync
  triggerSync: () => api.post('/sync'),

  // Risks
  getRisks: (params = {}) => {
    const queryParams = new URLSearchParams();
    if (params.risk_level) queryParams.append('risk_level', params.risk_level);
    if (params.status) queryParams.append('status', params.status);
    if (params.limit) queryParams.append('limit', params.limit);
    return api.get(`/risks?${queryParams.toString()}`);
  },
  getRiskDetail: (riskId) => api.get(`/risks/${riskId}`),
  submitReview: (riskId, action, notes) => 
    api.post(`/risks/${riskId}/review`, { risk_analysis_id: riskId, action, notes }),

  // Chatbot
  queryChatbot: (query) => api.post('/chatbot/query', { query }),

  // Audit Logs
  getAuditLogs: (params = {}) => {
    const queryParams = new URLSearchParams();
    if (params.action_type) queryParams.append('action_type', params.action_type);
    if (params.actor_email) queryParams.append('actor_email', params.actor_email);
    if (params.limit) queryParams.append('limit', params.limit);
    return api.get(`/audit/logs?${queryParams.toString()}`);
  },

  // Health
  healthCheck: () => api.get('/health'),
};

export default api;
