import React, { useState } from 'react';
import { useQuery } from '@tanstack/react-query';
import { apiClient } from '../lib/api';
import { DashboardLayout } from '../components/DashboardLayout';
import { ChatbotWidget } from '../components/ChatbotWidget';
import {
  FileText,
  Search,
  Filter,
  User,
  LogIn,
  UserPlus,
  RefreshCw,
  Eye,
  AlertTriangle,
  MessageSquare,
  MessageCircle
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '../components/ui/card';
import { Badge } from '../components/ui/badge';
import { Input } from '../components/ui/input';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '../components/ui/select';
import { ScrollArea } from '../components/ui/scroll-area';

const actionIcons = {
  USER_LOGIN: LogIn,
  USER_REGISTER: UserPlus,
  SYNC_INITIATED: RefreshCw,
  SYNC_COMPLETED: RefreshCw,
  CASE_REVIEWED: Eye,
  CASE_ESCALATED: AlertTriangle,
  CASE_DISMISSED: FileText,
  CHATBOT_QUERY: MessageSquare,
};

const actionColors = {
  USER_LOGIN: 'bg-blue-100 text-blue-700',
  USER_REGISTER: 'bg-green-100 text-green-700',
  SYNC_INITIATED: 'bg-purple-100 text-purple-700',
  SYNC_COMPLETED: 'bg-purple-100 text-purple-700',
  CASE_REVIEWED: 'bg-slate-100 text-slate-700',
  CASE_ESCALATED: 'bg-red-100 text-red-700',
  CASE_DISMISSED: 'bg-gray-100 text-gray-700',
  CHATBOT_QUERY: 'bg-amber-100 text-amber-700',
};

const AuditLogsPage = () => {
  const [filters, setFilters] = useState({
    action_type: '',
    actor_email: '',
  });

  const { data: logs, isLoading } = useQuery({
    queryKey: ['audit-logs', filters],
    queryFn: () =>
      apiClient
        .getAuditLogs({
          action_type: filters.action_type || undefined,
          actor_email: filters.actor_email || undefined,
          limit: 100,
        })
        .then((r) => r.data),
    refetchInterval: 30000,
  });

  const formatActionLabel = (action) => {
    return action
      .split('_')
      .map((word) => word.charAt(0) + word.slice(1).toLowerCase())
      .join(' ');
  };

  return (
    <DashboardLayout title="Audit Logs">
      {/* Filters */}
      <Card className="mb-6 border shadow-sm">
        <CardContent className="p-4">
          <div className="flex flex-wrap items-center gap-4">
            <div className="relative flex-1 min-w-[200px]">
              <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-slate-400" />
              <Input
                placeholder="Search by email..."
                value={filters.actor_email}
                onChange={(e) => setFilters((f) => ({ ...f, actor_email: e.target.value }))}
                className="pl-9"
                data-testid="audit-search"
              />
            </div>
            <Select
              value={filters.action_type}
              onValueChange={(value) => setFilters((f) => ({ ...f, action_type: value }))}
            >
              <SelectTrigger className="w-[180px]" data-testid="audit-action-filter">
                <Filter className="mr-2 h-4 w-4" />
                <SelectValue placeholder="Action Type" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Actions</SelectItem>
                <SelectItem value="USER_LOGIN">User Login</SelectItem>
                <SelectItem value="SYNC_COMPLETED">Sync Completed</SelectItem>
                <SelectItem value="CASE_REVIEWED">Case Reviewed</SelectItem>
                <SelectItem value="CASE_ESCALATED">Case Escalated</SelectItem>
                <SelectItem value="CHATBOT_QUERY">Chatbot Query</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Logs List */}
      <Card className="border shadow-sm">
        <CardHeader>
          <CardTitle className="font-['Manrope'] flex items-center gap-2">
            <FileText className="h-5 w-5 text-slate-500" />
            Activity Log
          </CardTitle>
        </CardHeader>
        <CardContent>
          <ScrollArea className="h-[600px]">
            {isLoading ? (
              <div className="flex items-center justify-center py-12">
                <div className="h-8 w-8 animate-spin rounded-full border-4 border-slate-200 border-t-blue-600"></div>
              </div>
            ) : logs && logs.length > 0 ? (
              <div className="space-y-3">
                {logs.map((log, index) => {
                  // Determine Icon and Color based on action
                  // Fallback to FileText if action is unknown
                  const normalizedAction = log.action.toUpperCase().replace(' ', '_');
                  const Icon = actionIcons[normalizedAction] || actionIcons[log.action] || FileText;
                  const colorClass = actionColors[normalizedAction] || actionColors[log.action] || 'bg-slate-100 text-slate-700';

                  // Destructure specific details we want to show specially
                  const { chat_topic, chat_participants, notes, risk_id, action: actionDetail, ...otherDetails } = log.details || {};

                  return (
                    <div
                      key={log.id || index}
                      className="flex items-start gap-4 p-4 rounded-lg border border-slate-200 hover:bg-slate-50 transition-colors"
                      data-testid={`audit-log-${index}`}
                    >
                      {/* Icon Circle */}
                      <div className={`flex h-10 w-10 items-center justify-center rounded-full shrink-0 ${colorClass}`}>
                        <Icon className="h-5 w-5" />
                      </div>

                      {/* Content Area */}
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center justify-between mb-1">
                          <Badge variant="outline" className={`${colorClass} font-semibold`}>
                            {formatActionLabel(log.action)}
                          </Badge>
                          <span className="text-xs text-slate-400">
                            {new Date(log.timestamp).toLocaleString()}
                          </span>
                        </div>
                        
                        <p className="text-sm font-medium text-slate-900 flex items-center gap-2 mb-2">
                          <User className="h-3.5 w-3.5 text-slate-400" />
                          {log.actor_email}
                        </p>

                        {/* --- NEW: Render Chat Context if available --- */}
                        {chat_topic && (
                          <div className="bg-white border border-slate-200 rounded-md p-2.5 mb-2 shadow-sm">
                            <div className="flex items-center gap-2 mb-1">
                                <MessageCircle className="h-3.5 w-3.5 text-blue-500" />
                                <span className="text-sm font-medium text-slate-800">
                                    Topic: {chat_topic}
                                </span>
                            </div>
                            {chat_participants && chat_participants.length > 0 && (
                                <p className="text-xs text-slate-500 pl-5.5">
                                    With: {Array.isArray(chat_participants) ? chat_participants.join(', ') : chat_participants}
                                </p>
                            )}
                          </div>
                        )}

                        {/* --- NEW: Render Reviewer Notes specially --- */}
                        {notes && (
                            <div className="text-sm text-slate-600 italic border-l-2 border-slate-300 pl-3 py-0.5 mb-2">
                                "{notes}"
                            </div>
                        )}

                        {/* Render Remaining Details (Generic Fallback) */}
                        {Object.keys(otherDetails).length > 0 && (
                          <div className="mt-2 text-xs text-slate-500 bg-slate-100/50 p-2 rounded">
                            {Object.entries(otherDetails).map(([key, value]) => (
                              <div key={key} className="flex gap-1">
                                <span className="font-semibold text-slate-600">{key.replace('_', ' ')}:</span>
                                <span className="truncate">{typeof value === 'object' ? JSON.stringify(value) : String(value)}</span>
                              </div>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>
                  );
                })}
              </div>
            ) : (
              <div className="py-12 text-center">
                <FileText className="mx-auto h-12 w-12 text-slate-300" />
                <h3 className="mt-4 text-lg font-medium text-slate-900">No audit logs</h3>
                <p className="mt-2 text-sm text-slate-500">
                  Activity will appear here when users interact with the system
                </p>
              </div>
            )}
          </ScrollArea>
        </CardContent>
      </Card>

      <ChatbotWidget />
    </DashboardLayout>
  );
};

export default AuditLogsPage;