import React from 'react';
import { useQuery } from '@tanstack/react-query';
import { apiClient } from '../lib/api';
import { DashboardLayout } from '../components/DashboardLayout';
import { ChatbotWidget } from '../components/ChatbotWidget';
import { Link } from 'react-router-dom';
import {
  AlertTriangle,
  AlertCircle,
  CheckCircle,
  TrendingUp,
  FileText,
  Clock,
  ArrowRight,
} from 'lucide-react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Badge } from '../components/ui/badge';
import { Button } from '../components/ui/button';
import {
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  Legend,
} from 'recharts';

const StatCard = ({ title, value, description, icon: Icon, variant = 'default' }) => {
  const variants = {
    default: 'bg-white',
    high: 'bg-red-50 border-red-200',
    medium: 'bg-amber-50 border-amber-200',
    low: 'bg-emerald-50 border-emerald-200',
  };

  const iconVariants = {
    default: 'text-slate-600 bg-slate-100',
    high: 'text-red-600 bg-red-100',
    medium: 'text-amber-600 bg-amber-100',
    low: 'text-emerald-600 bg-emerald-100',
  };

  return (
    <Card className={`${variants[variant]} border shadow-sm hover:shadow-md transition-shadow`}>
      <CardContent className="p-6">
        <div className="flex items-start justify-between">
          <div>
            <p className="text-sm font-medium text-slate-500">{title}</p>
            <p className="mt-2 text-3xl font-bold text-slate-900 font-['Manrope']">{value}</p>
            {description && <p className="mt-1 text-xs text-slate-500">{description}</p>}
          </div>
          <div className={`rounded-lg p-3 ${iconVariants[variant]}`}>
            <Icon className="h-5 w-5" />
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

const RiskBadge = ({ level }) => {
  const config = {
    HIGH: { className: 'bg-red-50 text-red-700 border-red-200', label: 'High Risk' },
    MEDIUM: { className: 'bg-amber-50 text-amber-700 border-amber-200', label: 'Medium Risk' },
    LOW: { className: 'bg-emerald-50 text-emerald-700 border-emerald-200', label: 'Low Risk' },
  };

  const { className, label } = config[level] || config.LOW;

  return (
    <Badge variant="outline" className={className}>
      {label}
    </Badge>
  );
};

const CustomTooltip = ({ active, payload, label }) => {
  if (active && payload && payload.length) {
    return (
      <div className="rounded-lg bg-slate-900 px-3 py-2 text-sm shadow-lg">
        <p className="text-slate-300">{label}</p>
        {payload.map((entry, index) => (
          <p key={index} style={{ color: entry.color }}>
            {entry.name}: {entry.value}
          </p>
        ))}
      </div>
    );
  }
  return null;
};

const DashboardPage = () => {
  const { data: stats, isLoading: statsLoading } = useQuery({
    queryKey: ['dashboard-stats'],
    queryFn: () => apiClient.getStats().then((r) => r.data),
    refetchInterval: 30000, // Refetch every 30 seconds
  });

  const { data: trends } = useQuery({
    queryKey: ['dashboard-trends'],
    queryFn: () => apiClient.getTrends().then((r) => r.data),
    refetchInterval: 60000,
  });

  const { data: recentAlerts } = useQuery({
    queryKey: ['recent-alerts'],
    queryFn: () => apiClient.getRecentAlerts(5).then((r) => r.data),
    refetchInterval: 30000,
  });

  const formattedTrends = trends?.map((t) => ({
    date: new Date(t.date).toLocaleDateString('en-US', { weekday: 'short' }),
    High: t.high,
    Medium: t.medium,
  }));

  return (
    <DashboardLayout title="Dashboard">
      {/* Stats Grid */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4 mb-6" data-testid="stats-grid">
        <StatCard
          title="High Risk Alerts"
          value={statsLoading ? '...' : stats?.high_risk_count || 0}
          description="Requires immediate attention"
          icon={AlertTriangle}
          variant="high"
        />
        <StatCard
          title="Medium Risk Alerts"
          value={statsLoading ? '...' : stats?.medium_risk_count || 0}
          description="Review recommended"
          icon={AlertCircle}
          variant="medium"
        />
        <StatCard
          title="Pending Reviews"
          value={statsLoading ? '...' : stats?.pending_reviews || 0}
          description="Awaiting compliance decision"
          icon={Clock}
          variant="default"
        />
        <StatCard
          title="Reviewed Today"
          value={statsLoading ? '...' : stats?.reviewed_today || 0}
          description="Cases processed"
          icon={CheckCircle}
          variant="low"
        />
      </div>

      <div className="grid gap-6 lg:grid-cols-3">
        {/* Risk Trends Chart */}
        <Card className="lg:col-span-2 border shadow-sm">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 font-['Manrope']">
              <TrendingUp className="h-5 w-5 text-slate-500" />
              Risk Trends (7 Days)
            </CardTitle>
            <CardDescription>Daily breakdown of flagged conversations</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-[300px]">
              {formattedTrends && formattedTrends.length > 0 ? (
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={formattedTrends}>
                    <CartesianGrid strokeDasharray="3 3" stroke="#E2E8F0" />
                    <XAxis dataKey="date" stroke="#64748B" fontSize={12} />
                    <YAxis stroke="#64748B" fontSize={12} />
                    <Tooltip content={<CustomTooltip />} />
                    <Legend />
                    <Line
                      type="monotone"
                      dataKey="High"
                      stroke="#EF4444"
                      strokeWidth={2}
                      dot={{ fill: '#EF4444' }}
                    />
                    <Line
                      type="monotone"
                      dataKey="Medium"
                      stroke="#F59E0B"
                      strokeWidth={2}
                      dot={{ fill: '#F59E0B' }}
                    />
                  </LineChart>
                </ResponsiveContainer>
              ) : (
                <div className="flex h-full items-center justify-center text-slate-500">
                  <div className="text-center">
                    <FileText className="mx-auto h-12 w-12 text-slate-300" />
                    <p className="mt-2">No trend data available</p>
                    <p className="text-sm">Click "Sync Teams" to fetch data</p>
                  </div>
                </div>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Recent Alerts */}
        <Card className="border shadow-sm">
          <CardHeader>
            <CardTitle className="flex items-center justify-between font-['Manrope']">
              <span className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-slate-500" />
                Recent Alerts
              </span>
              <Link to="/flagged">
                <Button variant="ghost" size="sm" className="text-blue-600">
                  View all
                  <ArrowRight className="ml-1 h-4 w-4" />
                </Button>
              </Link>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {recentAlerts && recentAlerts.length > 0 ? (
                recentAlerts.map((alert, index) => (
                  <Link
                    key={alert.id}
                    to={`/flagged/${alert.id}`}
                    className="block"
                    data-testid={`recent-alert-${index}`}
                  >
                    <div className="rounded-lg border border-slate-200 p-3 hover:bg-slate-50 transition-colors">
                      <div className="flex items-start justify-between">
                        <div className="min-w-0 flex-1">
                          <p className="truncate text-sm font-medium text-slate-900">
                            {alert.conversation?.participants
                              ?.map((p) => p.name || p.email)
                              .join(', ') || 'Unknown participants'}
                          </p>
                          <p className="mt-1 truncate text-xs text-slate-500">
                            {alert.risk_indicators?.slice(0, 2).join(', ') || 'Risk detected'}
                          </p>
                        </div>
                        <RiskBadge level={alert.risk_level} />
                      </div>
                    </div>
                  </Link>
                ))
              ) : (
                <div className="py-8 text-center text-slate-500">
                  <AlertCircle className="mx-auto h-8 w-8 text-slate-300" />
                  <p className="mt-2 text-sm">No alerts yet</p>
                  <p className="text-xs">Sync Teams to detect risks</p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      <ChatbotWidget />
    </DashboardLayout>
  );
};

export default DashboardPage;
