import React, { useState } from 'react';
import { useQuery } from '@tanstack/react-query';
import { useNavigate } from 'react-router-dom';
import { apiClient } from '../lib/api';
import { DashboardLayout } from '../components/DashboardLayout';
import { ChatbotWidget } from '../components/ChatbotWidget';
import {
  AlertTriangle,
  AlertCircle,
  Search,
  Filter,
  ChevronRight,
  Clock,
  CheckCircle,
  XCircle,
} from 'lucide-react';
import { Card, CardContent } from '../components/ui/card';
import { Badge } from '../components/ui/badge';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '../components/ui/select';

const RiskBadge = ({ level }) => {
  const config = {
    HIGH: {
      className: 'bg-red-50 text-red-700 border-red-200',
      icon: AlertTriangle,
      label: 'High Risk',
    },
    MEDIUM: {
      className: 'bg-amber-50 text-amber-700 border-amber-200',
      icon: AlertCircle,
      label: 'Medium Risk',
    },
    LOW: {
      className: 'bg-emerald-50 text-emerald-700 border-emerald-200',
      icon: CheckCircle,
      label: 'Low Risk',
    },
  };

  const { className, icon: Icon, label } = config[level] || config.LOW;

  return (
    <Badge variant="outline" className={`${className} flex items-center gap-1`}>
      <Icon className="h-3 w-3" />
      {label}
    </Badge>
  );
};

const StatusBadge = ({ status }) => {
  const config = {
    pending: { className: 'bg-slate-100 text-slate-700', icon: Clock, label: 'Pending' },
    reviewed: { className: 'bg-blue-50 text-blue-700', icon: CheckCircle, label: 'Reviewed' },
    escalated: { className: 'bg-red-50 text-red-700', icon: AlertTriangle, label: 'Escalated' },
    dismissed: { className: 'bg-slate-100 text-slate-500', icon: XCircle, label: 'Dismissed' },
  };

  const { className, icon: Icon, label } = config[status] || config.pending;

  return (
    <Badge variant="outline" className={`${className} flex items-center gap-1`}>
      <Icon className="h-3 w-3" />
      {label}
    </Badge>
  );
};

const FlaggedMessagesPage = () => {
  const navigate = useNavigate();
  const [filters, setFilters] = useState({
    risk_level: '',
    status: '',
    search: '',
  });

  const { data: risks, isLoading, refetch } = useQuery({
    queryKey: ['risks', filters],
    queryFn: () =>
      apiClient
        .getRisks({
          risk_level: filters.risk_level || undefined,
          status: filters.status || undefined,
          limit: 50,
        })
        .then((r) => r.data),
    refetchInterval: 30000,
  });

  const filteredRisks = risks?.filter((risk) => {
    if (!filters.search) return true;
    const searchLower = filters.search.toLowerCase();
    const participants = risk.conversation?.participants || [];
    return (
      participants.some(
        (p) =>
          p.name?.toLowerCase().includes(searchLower) ||
          p.email?.toLowerCase().includes(searchLower)
      ) ||
      risk.ai_explanation?.toLowerCase().includes(searchLower) ||
      risk.risk_indicators?.some((i) => i.toLowerCase().includes(searchLower))
    );
  });

  const handleRowClick = (riskId) => {
    navigate(`/flagged/${riskId}`);
  };

  return (
    <DashboardLayout title="Flagged Messages">
      {/* Filters */}
      <Card className="mb-6 border shadow-sm">
        <CardContent className="p-4">
          <div className="flex flex-wrap items-center gap-4">
            <div className="relative flex-1 min-w-[200px]">
              <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-slate-400" />
              <Input
                placeholder="Search by participant, keyword..."
                value={filters.search}
                onChange={(e) => setFilters((f) => ({ ...f, search: e.target.value }))}
                className="pl-9"
                data-testid="search-input"
              />
            </div>
            <Select
              value={filters.risk_level}
              onValueChange={(value) => setFilters((f) => ({ ...f, risk_level: value }))}
            >
              <SelectTrigger className="w-[150px]" data-testid="risk-level-filter">
                <Filter className="mr-2 h-4 w-4" />
                <SelectValue placeholder="Risk Level" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Levels</SelectItem>
                <SelectItem value="HIGH">High Risk</SelectItem>
                <SelectItem value="MEDIUM">Medium Risk</SelectItem>
              </SelectContent>
            </Select>
            <Select
              value={filters.status}
              onValueChange={(value) => setFilters((f) => ({ ...f, status: value }))}
            >
              <SelectTrigger className="w-[150px]" data-testid="status-filter">
                <Filter className="mr-2 h-4 w-4" />
                <SelectValue placeholder="Status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Status</SelectItem>
                <SelectItem value="pending">Pending</SelectItem>
                <SelectItem value="reviewed">Reviewed</SelectItem>
                <SelectItem value="escalated">Escalated</SelectItem>
                <SelectItem value="dismissed">Dismissed</SelectItem>
              </SelectContent>
            </Select>
            <Button
              variant="outline"
              onClick={() => {
                setFilters({ risk_level: '', status: '', search: '' });
                refetch();
              }}
              data-testid="clear-filters"
            >
              Clear Filters
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Results */}
      <div className="space-y-3">
        {isLoading ? (
          <div className="flex items-center justify-center py-12">
            <div className="h-8 w-8 animate-spin rounded-full border-4 border-slate-200 border-t-blue-600"></div>
          </div>
        ) : filteredRisks && filteredRisks.length > 0 ? (
          filteredRisks.map((risk, index) => (
            <Card
              key={risk.id}
              className="border shadow-sm hover:shadow-md transition-shadow cursor-pointer"
              onClick={() => handleRowClick(risk.id)}
              data-testid={`risk-row-${index}`}
            >
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-3 mb-2">
                      <RiskBadge level={risk.risk_level} />
                      <StatusBadge status={risk.status} />
                      <span className="text-xs text-slate-500">
                        Score: {risk.risk_score}/100
                      </span>
                    </div>
                    <h3 className="font-medium text-slate-900 truncate">
                      {risk.conversation?.participants?.map((p) => p.name || p.email).join(' ↔ ') ||
                        'Unknown Conversation'}
                    </h3>
                    <p className="mt-1 text-sm text-slate-600 line-clamp-2">
                      {risk.ai_explanation}
                    </p>
                    {risk.risk_indicators && risk.risk_indicators.length > 0 && (
                      <div className="mt-2 flex flex-wrap gap-1">
                        {risk.risk_indicators.slice(0, 3).map((indicator, i) => (
                          <span
                            key={i}
                            className="inline-flex items-center rounded-full bg-slate-100 px-2 py-0.5 text-xs text-slate-600"
                          >
                            {indicator}
                          </span>
                        ))}
                        {risk.risk_indicators.length > 3 && (
                          <span className="text-xs text-slate-400">
                            +{risk.risk_indicators.length - 3} more
                          </span>
                        )}
                      </div>
                    )}
                  </div>
                  <ChevronRight className="h-5 w-5 text-slate-400 flex-shrink-0 ml-4" />
                </div>
              </CardContent>
            </Card>
          ))
        ) : (
          <Card className="border shadow-sm">
            <CardContent className="py-12 text-center">
              <AlertCircle className="mx-auto h-12 w-12 text-slate-300" />
              <h3 className="mt-4 text-lg font-medium text-slate-900">No flagged messages</h3>
              <p className="mt-2 text-sm text-slate-500">
                {filters.search || filters.risk_level || filters.status
                  ? 'Try adjusting your filters'
                  : 'Click "Sync Teams" to fetch and analyze conversations'}
              </p>
            </CardContent>
          </Card>
        )}
      </div>

      <ChatbotWidget />
    </DashboardLayout>
  );
};

export default FlaggedMessagesPage;
