import React, { useState } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { apiClient } from '../lib/api';
import { toast } from 'sonner';
import { DashboardLayout } from '../components/DashboardLayout';
import { ChatbotWidget } from '../components/ChatbotWidget';
import {
  ArrowLeft,
  AlertTriangle,
  AlertCircle,
  CheckCircle,
  XCircle,
  Clock,
  Send,
  Eye,
  Flag,
  User,
  Loader2,
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '../components/ui/card';
import { Badge } from '../components/ui/badge';
import { Button } from '../components/ui/button';
import { Textarea } from '../components/ui/textarea';
import { ScrollArea } from '../components/ui/scroll-area';
import { Avatar, AvatarFallback } from '../components/ui/avatar';
import { cn } from '../lib/utils';

const RiskBadge = ({ level, large = false }) => {
  const config = {
    HIGH: {
      className: 'bg-red-50 text-red-700 border-red-200',
      icon: AlertTriangle,
      label: 'High Risk',
    },
    MEDIUM: {
      className: 'bg-amber-50 text-amber-700 border-amber-200',
      icon: AlertCircle,
      label: 'Medium Risk',
    },
    LOW: {
      className: 'bg-emerald-50 text-emerald-700 border-emerald-200',
      icon: CheckCircle,
      label: 'Low Risk',
    },
  };

  const { className, icon: Icon, label } = config[level] || config.LOW;

  return (
    <Badge
      variant="outline"
      className={cn(
        className,
        'flex items-center gap-1',
        large && 'px-3 py-1 text-sm'
      )}
    >
      <Icon className={large ? 'h-4 w-4' : 'h-3 w-3'} />
      {label}
    </Badge>
  );
};

const ChatBubble = ({ message, isOutgoing, isFlagged }) => {
  const initials = message.sender_name
    ?.split(' ')
    .map((n) => n[0])
    .join('')
    .toUpperCase()
    .slice(0, 2) || 'U';

  return (
    <div
      className={cn(
        'flex gap-3 mb-4',
        isOutgoing ? 'flex-row-reverse' : 'flex-row'
      )}
    >
      <Avatar className="h-8 w-8 flex-shrink-0">
        <AvatarFallback className={isOutgoing ? 'bg-blue-100 text-blue-700' : 'bg-slate-100 text-slate-700'}>
          {initials}
        </AvatarFallback>
      </Avatar>
      <div className={cn('max-w-[70%]', isOutgoing && 'text-right')}>
        <div className="flex items-center gap-2 mb-1">
          <span className="text-xs font-medium text-slate-700">{message.sender_name}</span>
          <span className="text-xs text-slate-400">
            {new Date(message.timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
          </span>
        </div>
        <div
          className={cn(
            'rounded-lg px-4 py-2 shadow-sm',
            isOutgoing
              ? 'bg-blue-50 border border-blue-200 text-slate-700'
              : 'bg-white border border-slate-200 text-slate-700',
            isFlagged && 'bg-red-50 border-2 border-red-300'
          )}
        >
          <p className="text-sm whitespace-pre-wrap">{message.content}</p>
        </div>
      </div>
    </div>
  );
};

const MessageReviewPage = () => {
  const { riskId } = useParams();
  const navigate = useNavigate();
  const queryClient = useQueryClient();
  const [notes, setNotes] = useState('');

  const { data: risk, isLoading } = useQuery({
    queryKey: ['risk-detail', riskId],
    queryFn: () => apiClient.getRiskDetail(riskId).then((r) => r.data),
  });

  const reviewMutation = useMutation({
    mutationFn: ({ action, notes }) => apiClient.submitReview(riskId, action, notes),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['risk-detail', riskId] });
      queryClient.invalidateQueries({ queryKey: ['risks'] });
      queryClient.invalidateQueries({ queryKey: ['dashboard-stats'] });
      toast.success('Review submitted successfully');
    },
    onError: (error) => {
      toast.error(error.response?.data?.detail || 'Failed to submit review');
    },
  });

  const handleAction = (action) => {
    reviewMutation.mutate({ action, notes });
    setNotes('');
  };

  if (isLoading) {
    return (
      <DashboardLayout title="Message Review">
        <div className="flex items-center justify-center py-12">
          <Loader2 className="h-8 w-8 animate-spin text-slate-600" />
        </div>
      </DashboardLayout>
    );
  }

  if (!risk) {
    return (
      <DashboardLayout title="Message Review">
        <Card className="border shadow-sm">
          <CardContent className="py-12 text-center">
            <AlertCircle className="mx-auto h-12 w-12 text-slate-300" />
            <h3 className="mt-4 text-lg font-medium text-slate-900">Risk not found</h3>
            <Button onClick={() => navigate('/flagged')} className="mt-4">
              Back to Flagged Messages
            </Button>
          </CardContent>
        </Card>
      </DashboardLayout>
    );
  }

  const messages = risk.conversation?.messages || [];
  const participants = risk.conversation?.participants || [];
  const primaryParticipant = participants[0]?.email || '';

  return (
    <DashboardLayout title="Message Review">
      {/* Back Button */}
      <Button
        variant="ghost"
        onClick={() => navigate('/flagged')}
        className="mb-4"
        data-testid="back-button"
      >
        <ArrowLeft className="mr-2 h-4 w-4" />
        Back to Flagged Messages
      </Button>

      <div className="grid gap-6 lg:grid-cols-5">
        {/* Chat View - Left Pane */}
        <Card className="lg:col-span-3 border shadow-sm">
          <CardHeader className="border-b">
            <div className="flex items-center justify-between">
              <CardTitle className="font-['Manrope'] flex items-center gap-2">
                <User className="h-5 w-5 text-slate-500" />
                Conversation
              </CardTitle>
              <RiskBadge level={risk.risk_level} large />
            </div>
            <div className="flex flex-wrap gap-2 mt-2">
              {participants.map((p, i) => (
                <Badge key={i} variant="outline" className="text-xs">
                  {p.name || p.email}
                </Badge>
              ))}
            </div>
          </CardHeader>
          <CardContent className="p-0">
            <ScrollArea className="h-[500px] p-4">
              {messages.length > 0 ? (
                messages.map((message, index) => (
                  <ChatBubble
                    key={message.id || index}
                    message={message}
                    isOutgoing={message.sender_email === primaryParticipant}
                    isFlagged={risk.risk_level === 'HIGH'}
                  />
                ))
              ) : (
                <div className="flex items-center justify-center h-full text-slate-500">
                  No messages available
                </div>
              )}
            </ScrollArea>
          </CardContent>
        </Card>

        {/* Compliance Workspace - Right Pane */}
        <div className="lg:col-span-2 space-y-6">
          {/* AI Insight Card */}
          <Card
            className={cn(
              'border shadow-sm',
              risk.risk_level === 'HIGH' && 'border-red-200 bg-red-50/50'
            )}
          >
            <CardHeader>
              <CardTitle className="font-['Manrope'] flex items-center gap-2 text-lg">
                <AlertTriangle
                  className={cn(
                    'h-5 w-5',
                    risk.risk_level === 'HIGH' ? 'text-red-600' : 'text-amber-600'
                  )}
                />
                AI Risk Analysis
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium text-slate-500">Risk Score</span>
                <span className="text-2xl font-bold text-slate-900">{risk.risk_score}/100</span>
              </div>

              <div>
                <h4 className="text-sm font-medium text-slate-500 mb-2">Explanation</h4>
                <p className="text-sm text-slate-700 leading-relaxed">{risk.ai_explanation}</p>
              </div>

              {risk.risk_indicators && risk.risk_indicators.length > 0 && (
                <div>
                  <h4 className="text-sm font-medium text-slate-500 mb-2">Risk Indicators</h4>
                  <div className="flex flex-wrap gap-2">
                    {risk.risk_indicators.map((indicator, i) => (
                      <Badge
                        key={i}
                        variant="outline"
                        className="bg-slate-100 text-slate-700 border-slate-200"
                      >
                        {indicator}
                      </Badge>
                    ))}
                  </div>
                </div>
              )}

              {risk.cross_chat_context && (
                <div>
                  <h4 className="text-sm font-medium text-slate-500 mb-2">Cross-Chat Context</h4>
                  <p className="text-xs text-slate-600 bg-slate-100 p-3 rounded-lg">
                    {risk.cross_chat_context}
                  </p>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Action Buttons */}
          <Card className="border shadow-sm">
            <CardHeader>
              <CardTitle className="font-['Manrope'] text-lg">Review Actions</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <label className="text-sm font-medium text-slate-500 mb-2 block">
                  Notes (optional)
                </label>
                <Textarea
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  placeholder="Add notes about your decision..."
                  className="min-h-[100px]"
                  data-testid="review-notes"
                />
              </div>

              <div className="grid grid-cols-3 gap-2">
                <Button
                  variant="outline"
                  className="border-slate-200 hover:bg-slate-50"
                  onClick={() => handleAction('Dismiss')}
                  disabled={reviewMutation.isPending}
                  data-testid="action-dismiss"
                >
                  <XCircle className="mr-1 h-4 w-4 text-slate-500" />
                  Dismiss
                </Button>
                <Button
                  variant="outline"
                  className="border-blue-200 text-blue-700 hover:bg-blue-50"
                  onClick={() => handleAction('Monitor')}
                  disabled={reviewMutation.isPending}
                  data-testid="action-monitor"
                >
                  <Eye className="mr-1 h-4 w-4" />
                  Monitor
                </Button>
                <Button
                  className="bg-red-600 hover:bg-red-700 text-white"
                  onClick={() => handleAction('Escalate')}
                  disabled={reviewMutation.isPending}
                  data-testid="action-escalate"
                >
                  <Flag className="mr-1 h-4 w-4" />
                  Escalate
                </Button>
              </div>

              {risk.status !== 'pending' && (
                <div className="pt-2 border-t">
                  <p className="text-xs text-slate-500">
                    Current Status:{' '}
                    <span className="font-medium capitalize">{risk.status}</span>
                  </p>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Review History */}
          {risk.review_actions && risk.review_actions.length > 0 && (
            <Card className="border shadow-sm">
              <CardHeader>
                <CardTitle className="font-['Manrope'] text-lg">Review History</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {risk.review_actions.map((action, i) => (
                    <div
                      key={i}
                      className="flex items-start gap-3 pb-3 border-b last:border-0 last:pb-0"
                    >
                      <div className="flex h-8 w-8 items-center justify-center rounded-full bg-slate-100">
                        <User className="h-4 w-4 text-slate-600" />
                      </div>
                      <div className="flex-1 min-w-0">
                        <p className="text-sm font-medium text-slate-900">
                          {action.reviewer_name}
                        </p>
                        <p className="text-xs text-slate-500">
                          {action.action} •{' '}
                          {new Date(action.timestamp).toLocaleString()}
                        </p>
                        {action.notes && (
                          <p className="mt-1 text-xs text-slate-600 bg-slate-50 p-2 rounded">
                            {action.notes}
                          </p>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>

      <ChatbotWidget />
    </DashboardLayout>
  );
};

export default MessageReviewPage;
